rm(list=ls())
#' 9017 Online Panels Benchmarking Study SRM Paper
#' Analysis syntax
#'
#' @author  Dina Neiger
#' @version 20220112
#'
#'
#'
#' @input   
#' Sigtest file from Step 3
#' Benchmark data (inputs directory)
#' 
#' 
#' @output
#' Significance tests for comparison of unweighted estimates with benchmark (Table C2) and pairwise comparisons  (Table C4)
#' 
#' 
#' This syntax is structured in 3 parts:
#' ############Part 1: Calculation of test statistics
#' Two types of test statistics are calculated:
#'       - modal % for substantive variables 
#'       - absolute error (difference between the modal value and the corresponding benchmark)
#'       - final post-stratification weight is used (weight1)
#'      
#'  
#'  #############Part 2: Using bootstrap algorithm to calculate standard errors for the test statistics
#'  This method is used to ensure consistent method for standard error calculations across
#'  probability and non-probability surveys
#'  
#'  #############Part 3: Significance testing
#'  3a) Use t-test to  compare each substantive measure's modal response with the corresponding benchmark
#'      Census benchmarks are treated as true values
#'      Standard Errors for the survey benchmarks sourced from the respective sources
#'      
#'  3b) Use t-test to undertake pairwise comparisons of absolute errors between 8 surveys 
#'  
#'  The methodology is based on Yeager et al "Public Opinion Quarterly" V75, No4, pp709-747 and online supplement
#'      
#' 



options(java.parameters = "-Xmx6096m")


library("plyr")
library("xlsx")
library(reshape2)
library("foreign")

#install.packages("survey")
#install.packages("boot")

library("survey")
library("boot")

# Date formats for xlsxssf
options(xlsx.date.format="dd/MM/yyyy")
options(xlsx.datetime.format="dd/MM/yyyy")
# Windows stub


# Windows stub
Z_PATH <- "Z:/"

setwd("Z:/Research Papers and Presentations/SRM article/Submission syntax")

WRK_DIR <- setwd("Z:/Research Papers and Presentations/SRM article/Submission syntax")

INP_DIR <- paste0(WRK_DIR,"/Inputs/")
OUT_DIR <- paste0(WRK_DIR, "/Outputs/")


#load sigdata dataset
load(paste0(OUT_DIR, "S3-udata4sigtest.RData"))

#load varnames
load(paste0(OUT_DIR, "S2-variable names for sig testing.RData"))

#read benchmarks
load(paste0(INP_DIR, "bmark_data.RData"))

table(is.na(sigdata_unweighted$b1),exclude=NULL)
table(is.na(sigdata_unweighted$k6cat),exclude=NULL)
table(is.na(sigdata_unweighted$b2),exclude=NULL)
table(is.na(sigdata_unweighted$d15),exclude=NULL)
table(is.na(sigdata_unweighted$b4),exclude=NULL)
table(is.na(sigdata_unweighted$b5),exclude=NULL)

#function for calculation of statistic and absolute error calculation to be used in the boot (resampling) function
stat_fn<-function(data,var, bmrk, indices){
  d<-data[indices,]
  d<-d[!is.na(d[,var]),]
  stat   <- 100*sum(d[,var])/nrow(d) 
  abserr <- abs(stat-bmrk)   
  return(c(stat,abserr))
}

#checking code for estimate calculation
stat_fn(sigdata_unweighted[sigdata_unweighted$surtype.l=="2ABS",],"b1_8w1",bmark_data[1,"bmrk"])

#checking code for bootstrap function
   set.seed(987)
   x1 <- boot(data=sigdata_unweighted[sigdata_unweighted$surtype.l=="2ABS",], R=1000,
                  statistic=stat_fn,var="b1_8w1",bmrk=bmark_data[1,"bmrk"])
   print(x1)
   sd(x1$t[,1])
   sd(x1$t[,2])

#create dataset of estimates, std errors, abserrors for each variable and each survey
nrow(varnames_ds)
est_data <- NULL
rown<-1
set.seed(987)
for(i in 1:nrow(varnames_ds)){
  for(j in 1:length(unique(sigdata_unweighted$surtype))){
   est_data$varname[rown]    <- varnames_ds$varname[i]
   est_data$wvarname[rown]   <- varnames_ds$wvarname[i]
   est_data$surtype[rown]    <- j
   est_data$stat[rown]       <- stat_fn(sigdata_unweighted[sigdata_unweighted$surtype==j,],varnames_ds$wvarname[i],bmark_data[i,"bmrk"])[1]
   est_data$abserr[rown]     <- stat_fn(sigdata_unweighted[sigdata_unweighted$surtype==j,],varnames_ds$wvarname[i],bmark_data[i,"bmrk"])[2]
   secalc                    <- boot(data=sigdata_unweighted[sigdata_unweighted$surtype==j,], R=1000,
                                      statistic=stat_fn,var=varnames_ds$wvarname[i],
                                      bmrk=bmark_data[i,"bmrk"])
   est_data$stat_se[rown]   <- sd(secalc$t[,1])
   est_data$stat_base[rown] <- nrow(sigdata_unweighted[sigdata_unweighted$surtype==j,])
   est_data$abserr_se[rown] <- sd(secalc$t[,2])
   rown                     <- rown+1
   
  }
  est_data$rank[est_data$wvarname %in% varnames_ds$wvarname[i]]<-rank(est_data$abserr[est_data$wvarname %in% varnames_ds$wvarname[i]])
  est_data$ave_abserr[est_data$wvarname %in% varnames_ds$wvarname[i]]<-ave(est_data$abserr[est_data$wvarname %in% varnames_ds$wvarname[i]])
  est_data$sd_abserr[est_data$wvarname %in% varnames_ds$wvarname[i]] <-sd(est_data$abserr[est_data$wvarname %in% varnames_ds$wvarname[i]])
}
est_data<-as.data.frame(est_data,stringsAsFactors=F)
#merge in surtype lable and benchmark data for calculation of significance
est_data <- merge(est_data,unique(sigdata_unweighted[,c("surtype","surtype.l")],by="surtype"))
est_data <- merge(est_data,bmark_data,by="varname")

#calculate t-test for benchmark comparison
est_data$S         <- sqrt((est_data$stat_base*(est_data$stat_base-1)* est_data$stat_se*est_data$stat_se+
                            est_data$bmrk_base*(est_data$bmrk_base-1)* est_data$bmrk_se*est_data$bmrk_se)/
                             (est_data$stat_base+est_data$bmrk_base-2))
est_data$t.value   <- (est_data$stat-est_data$bmrk)/(est_data$S*sqrt(1/est_data$stat_base + 1/est_data$bmrk_base))
est_data$p.value   <- round(2*pt(-abs(est_data$t.value),df=(est_data$stat_base+est_data$bmrk_base-2)),5)
est_data$notsig05  <-""
est_data$notsig05[est_data$p.value>=0.05] <-"yes"
est_data$notsig01  <-""
est_data$notsig01[est_data$p.value>=0.01] <-"yes"

    
est_data <- as.data.frame(est_data,stringsAsFactors=F)[with(est_data, order(varlabel, surtype.l)),]
est_data_out <- est_data[,c("varlabel","varname","catname","surtype.l","stat_base","stat","bmrk",
                            "notsig05","notsig01","p.value",
                            "abserr","rank","ave_abserr","sd_abserr",
                            "bmrk_base","stat_se","bmrk_se","S","t.value","surtype","abserr_se","wvarname")]
    

chn1 <- names(est_data) %in% names(est_data_out)
names(est_data)[!chn1]

#summary by survey
survey_summary <- ddply(est_data_out,.(surtype.l),summarize,
                        ave_abserr = sum(abserr)/length(abserr),
                        no_notsig05  = sum(notsig05=="yes"),
                        no_notsig01  = sum(notsig01=="yes"),
                        no_r1      = sum(rank==1),
                        no_r2      = sum(rank==2),
                        no_r3      = sum(rank==3),
                        no_r6      = sum(rank==6),
                        no_r7      = sum(rank==7),
                        no_r8      = sum(rank==8)
                        )



#do pair wise comparisons of the average abs error
#function for calculation of statistic and absolute error calculation to be used in the boot (resampling) function
aerr_fn<-function(data, indices){
  d<-data[indices,]
  aved <- NULL
  for(i in 1:nrow(varnames_ds)){
    aved[i]<-stat_fn(d,varnames_ds$wvarname[i],bmark_data[i,"bmrk"])[2]
  }
  return(mean(aved))
  
}
#checking code for estimate calculation
aerr_fn(sigdata_unweighted[sigdata_unweighted$surtype.l %in% "2ABS",])
#calculate standard errors for average abs error

aerr_data <- NULL
set.seed(987)
for(j in 1:length(unique(sigdata_unweighted$surtype))){
    aerr_data$surtype[j]      <- j
    aerr_data$surtype.l[j]<- unique(sigdata_unweighted$surtype.l[sigdata_unweighted$surtype==j])
    aerr_data$base[j]         <- nrow(sigdata_unweighted[sigdata_unweighted$surtype==j,])
    aerr_data$aerr_stat[j]    <- aerr_fn(sigdata_unweighted[sigdata_unweighted$surtype %in% j,])
    aerr_secalc               <- boot(data=sigdata_unweighted[sigdata_unweighted$surtype %in% j,], R=1000,statistic=aerr_fn)
    aerr_data$aerr_se[j]      <- sd(aerr_secalc$t[,1])
}
aerr_data <- as.data.frame(aerr_data,stringsAsFactors=F)[with(aerr_data, order(surtype.l)),]

#pairwise significance testing
pairw_m <- matrix("", nrow=length(unique(sigdata_unweighted$surtype))+1,ncol=length(unique(sigdata_unweighted$surtype)))
colnames(pairw_m) <- unique(aerr_data$surtype.l)
rownames(pairw_m) <- c("ave error",unique(aerr_data$surtype.l))
pairw_m

for(i in 1:nrow(pairw_m)){
  if(i==1) pairw_m[1,]<-round(aerr_data$aerr_stat,2)
  else{
  rows <- i-1
  for(cols in 1:ncol(pairw_m)){
       #calculate the difference between average errors
       pairw_m[i,cols] <- round(aerr_data$aerr_stat[rows]-aerr_data$aerr_stat[cols],2)
       #calculate p-value
       S         <- sqrt((aerr_data$base[rows]*(aerr_data$base[rows]-1)*aerr_data$aerr_se[rows]*aerr_data$aerr_se[rows]+
                          aerr_data$base[cols]*(aerr_data$base[cols]-1)*aerr_data$aerr_se[cols]*aerr_data$aerr_se[cols])/
                         (aerr_data$base[rows]+aerr_data$base[cols]-2)
                         )
       t.value   <- (aerr_data$aerr_stat[rows]-aerr_data$aerr_stat[cols])/
                    (S*sqrt(1/aerr_data$base[rows] + 1/aerr_data$base[cols]))
       p.value   <- round(2*pt(-abs(t.value),df=(aerr_data$base[rows]+aerr_data$base[cols]-2)),5)
       #add significance symbol +p<.10, *p<.05, **p<.01, ***p<.001
       sig.sym   <-""
       if(p.value<0.10) sig.sym  <-"+"
       if(p.value<0.05) sig.sym  <-"*"
       if(p.value<0.01) sig.sym  <-"**"
       if(p.value<0.001) sig.sym <-"***"
       pairw_m[i,cols] <- paste(pairw_m[i,cols],sig.sym)
       if(rows==cols) pairw_m[i,cols]="-"
     }
}
}
pairw_m

#save benchmark comparisons
write.xlsx2(survey_summary,paste(OUT_DIR,"benchmark_comp_sigtest_unweighted.xlsx", sep=""),sheetName="survey summary", append=FALSE,row.names = FALSE,showNA=FALSE)
write.xlsx2(est_data_out,paste(OUT_DIR,"benchmark_comp_sigtest_unweighted.xlsx", sep=""),sheetName="var by survey",append=TRUE, row.names = FALSE,showNA=FALSE)

#save pairwise comparisons
write.xlsx2(pairw_m,paste(OUT_DIR,"benchmark_comp_sigtest_unweighted.xlsx", sep=""),sheetName="pairwise comp",append=TRUE, row.names = TRUE,showNA=FALSE)
write.xlsx2(aerr_data,paste(WRK_DIR,"benchmark_comp_sigtest_unweighted.xlsx", sep=""),sheetName="pairwise data",append=TRUE, row.names = FALSE,showNA=FALSE)

#save workspace
